/*
 * 
 * copywrite 2003 under GNU/GPL which means what it means.
 * Edscott Wilson Garcia 
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */



#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <limits.h>

#include <dirent.h>
#include <fcntl.h>
#include <signal.h>
#include <time.h>
#include <utime.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include "glade_callbacks.h"
#include "glade_gui.h"
#include "glade_support.h"

#include "constants.h"
#include "types.h"

#include "add_file.h"
#include "cpy.h"
#include "entry.h"
#include "misc.h"
#include "monitor.h"
#include "remove.h"
#include "tubo.h"
#include "uri.h"
#include "widgets.h"


#define MAX_LINE_SIZE (_POSIX_PATH_MAX*3)
#define CHILD_FILE_LENGTH 64

#ifndef GLOB_TILDE
#define GLOB_TILDE 0x0
#endif

#ifndef GLOB_PERIOD
#define GLOB_PERIOD 0
#endif

extern int stop;

static GtkTreeView *cpy_treeview;
static gboolean waste = TRUE;

gboolean tar_extraction = FALSE;
char *src_host = NULL;
static char child_file[_POSIX_PATH_MAX];
static int child_mode;

static char *fork_target, *fork_source;
static gboolean I_am_child = FALSE;
static char *holdfile;
static char *targetdir;
static char dlg_result;
static gboolean force_override = FALSE;
static gboolean same_device;

static int nitems;

static GtkWidget *count_label;
static gboolean count_cancelled;
static GtkWidget *countW = NULL;
static int counttimer, countT, smallcount;
static long long sizeT;

static gboolean incomplete_target = FALSE;
static int child_path_number;
static gboolean all_recursive = FALSE;


static void *rw_fork_obj;
time_t initial;


static void close_it(tree_details_t * tree_details)
{
    gtk_main_quit();
    gtk_widget_destroy(tree_details->remove);
    tree_details->remove = NULL;
}

static gboolean on_destroy_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
    GtkTreeView *treeview = (GtkTreeView *) user_data;
    tree_details_t *tree_details = get_tree_details(treeview);
    dlg_result = DLG_CANCEL;
    close_it(tree_details);
    return TRUE;
}

#if 0
static void on_copy_stop_activate(GtkButton * button, gpointer user_data)
{
    GtkTreeView *treeview = (GtkTreeView *) user_data;
    tree_details_t *tree_details = get_tree_details(treeview);
    dlg_result = DLG_CANCEL;
    close_it(tree_details);

}
#endif

static void on_copy_no_activate(GtkButton * button, gpointer user_data)
{
    GtkWidget *check;
    GtkTreeView *treeview = (GtkTreeView *) user_data;
    tree_details_t *tree_details = get_tree_details(treeview);
    dlg_result = DLG_NO;
    check = lookup_widget(tree_details->remove, "togglebutton1");
    force_override = gtk_toggle_button_get_active((GtkToggleButton *) check);
    if (force_override) dlg_result = DLG_CANCEL;
    close_it(tree_details);
}

static void on_copy_yes_activate(GtkButton * button, gpointer user_data)
{
    GtkWidget *check;
    GtkTreeView *treeview = (GtkTreeView *) user_data;
    tree_details_t *tree_details = get_tree_details(treeview);
    dlg_result = DLG_YES;
    check = lookup_widget(tree_details->remove, "togglebutton1");
    force_override = gtk_toggle_button_get_active((GtkToggleButton *) check);
    check = lookup_widget(tree_details->remove, "radiobutton2");
    if(gtk_toggle_button_get_active((GtkToggleButton *) check))
	waste = FALSE;

    close_it(tree_details);
}


static int make_dialog(GtkTreeView * treeview, char *err, char *target)
{
    tree_details_t *tree_details = get_tree_details(treeview);
    if(tree_details->remove)
	g_assert_not_reached();

    waste = TRUE;
    tree_details->remove = create_remove();
    gtk_window_set_position(GTK_WINDOW(tree_details->remove), GTK_WIN_POS_MOUSE);

/*  g_signal_connect_object (lookup_widget (tree_details->remove, "button203"),
			   "clicked",
			   G_CALLBACK (on_copy_stop_activate),
			   (gpointer) treeview, 0);*/
    g_signal_connect_object(lookup_widget(tree_details->remove, "button204"), "clicked", G_CALLBACK(on_copy_no_activate), (gpointer) treeview, 0);
    g_signal_connect_object(lookup_widget(tree_details->remove, "button224"), "clicked", G_CALLBACK(on_copy_yes_activate), (gpointer) treeview, 0);
/*  g_signal_connect_object ( 
		  lookup_widget(tree_details->remove,"togglebutton1"),
		  "toggled",
                  G_CALLBACK (on_copy_all_toggled),
                  (gpointer)treeview, 0);*/

    g_signal_connect_object(tree_details->remove, "delete-event", G_CALLBACK(on_destroy_event), (gpointer) treeview, 0);
    g_signal_connect_object(tree_details->remove, "destroy-event", G_CALLBACK(on_destroy_event), (gpointer) treeview, 0);

    /* dialog specifics */
    /* no real use schreding before overwriting */
    hideit(tree_details->remove, "radiobutton3");

    if(err)
    {
	char mess[_POSIX_PATH_MAX];
	sprintf(mess, "%s (%s)? ", _("Try again"), err);
	gtk_label_set_text((GtkLabel *) lookup_widget(tree_details->remove, "label16"), mess);
	hideit(tree_details->remove, "question");
    }
    else
    {
	gtk_label_set_text((GtkLabel *) lookup_widget(tree_details->remove, "label16"), _("Overwrite"));
	hideit(tree_details->remove, "warning");
    }

    gtk_label_set_text((GtkLabel *) 
		    lookup_widget(tree_details->remove, "label20"),  
		    my_cut_utf_string(target) );
    gtk_widget_realize(tree_details->remove);
    gtk_widget_show(tree_details->remove);

    gtk_window_set_transient_for(GTK_WINDOW(tree_details->remove), 
		    GTK_WINDOW(tree_details->window));
    
    if (strstr(target,"/..Wastebasket"))    {
	hideit(tree_details->remove, "radiobutton1");
	gtk_toggle_button_set_active((GtkToggleButton *) 
			lookup_widget(tree_details->remove, "radiobutton2"), 
			TRUE);
    }

    gtk_main();
    return dlg_result;
}





/**********************************************************************/

/*
 * process_error sends the appropriate error text from
 * child process to parent  */
/* byte 1 */
#define RW_ERROR_OPENING_SRC	0x02
#define RW_ERROR_OPENING_TGT	0x04
#define RW_ERROR_TOO_FEW	0x08
#define RW_ERROR_TOO_MANY	0x10
#define RW_ERROR_READING_SRC	0x20
#define RW_ERROR_WRITING_TGT	0x40
#define RW_ERROR_STAT_READ_SRC	0x80
/* byte 2 */
#define RW_ERROR_STAT_READ_TGT	0x100
#define RW_ERROR_STAT_WRITE_TGT	0x200
#define RW_ERROR_MULTIPLE2FILE	0x400
#define RW_ERROR_DIR2FILE	0x800
#define RW_ERROR_ENTRY_NEW	0x1000
#define RW_ERROR_CLOSE_SRC	0x2000
#define RW_ERROR_CLOSE_TGT	0x4000
#define RW_ERROR_UNLINK_SRC	0x8000
/* byte 3 */
#define RW_ERRNO		0x10000
#define RW_ERROR_FIFO		0x20000
#define RW_ERROR_DEVICE		0x40000
#define RW_ERROR_SOCKET		0x80000
#define RW_ERROR_WRITING_SRC	0x100000
#define RW_OK			0x200000
#define RW_ERROR_WRITING_DIR 	0x400000

static int process_error(char *who, int code)
{
    char *message = "runOver", *txt = NULL;
    gboolean runover = FALSE;
    switch (code)
    {
	case RW_ERRNO:
	    message = strerror(errno);
	    break;
	case RW_ERROR_WRITING_DIR:
	    txt = targetdir;
	    message = strerror(errno);
	    break;
	case RW_ERROR_READING_SRC:
	case RW_ERROR_OPENING_SRC:
	case RW_ERROR_WRITING_SRC:
	    txt = fork_source;
	    message = strerror(errno);
	    break;
	case RW_ERROR_STAT_WRITE_TGT:
	case RW_ERROR_WRITING_TGT:
	case RW_ERROR_OPENING_TGT:
	    txt = fork_target;
	    message = strerror(errno);
	    /*return code; */
	    break;
	case RW_ERROR_TOO_FEW:
	    txt = fork_target;
	    message = _("Too few bytes transferred ! Device full ?");
	    break;
	case RW_ERROR_TOO_MANY:
	    txt = fork_target;
	    message = _("Too many bytes transferred !?");
	    break;
	case RW_ERROR_STAT_READ_SRC:
	    txt = fork_source;
	    message = _("Can't stat() file");
	    break;
	case RW_ERROR_STAT_READ_TGT:
	    txt = fork_source;
	    message = _("Can't stat() file");
	    break;
	case RW_ERROR_CLOSE_TGT:
	    txt = fork_target;
	    message = _("Error closing file");
	    break;
	case RW_ERROR_CLOSE_SRC:
	    txt = fork_source;
	    message = _("Error closing file");
	    break;
	case RW_ERROR_UNLINK_SRC:
	    txt = fork_source;
	    message = _("Error deleting");
	    break;
	case RW_ERROR_FIFO:
	    txt = fork_source;
	    message = _("Can't copy FIFO");
	    break;
	case RW_ERROR_DEVICE:
	    txt = fork_source;
	    message = _("Can't copy device file");
	    break;
	case RW_ERROR_SOCKET:
	    txt = fork_source;
	    message = _("Can't copy socket");
	    break;
	default:
	    runover = TRUE;
	    break;
    }
    if(strstr(message, "\n"))
	strtok(message, "\n");

    printf("child:%s: %s %s\n", who, message, txt);
    fflush(NULL);
    return code;
}

int rsync(GtkTreeView * treeview, char **srcs, char *tgt)
{
    char *argv[1024 + 50], *c, *cmd;
    int length, j, argc = 0;
    char *term="xterm", *all_srcs;
    int status;
    tree_details_t *tree_details = get_tree_details(treeview);
    pid_t pid;


    if ( getenv("TERM") && strlen( getenv("TERM"))){
	    gchar *c=g_find_program_in_path(getenv("TERM"));
	    if (c && access(c,X_OK)==0){
	      term = getenv("TERM");
	    }
	    g_free(c);
    }
	      
    if(!term || strlen(term) == 0)
	term = "xterm";
    argv[argc++] = term;
    if(strcmp(term, "xterm") == 0)
    {
	argv[argc++] = "+mesg";
	argv[argc++] = "-aw";
	argv[argc++] = "+cm";
	argv[argc++] = "+dc";
	if (getenv("XFFM_HOLD_XTERM") && strlen( getenv("XFFM_HOLD_XTERM")))
	       	argv[argc++] = "-hold";
	argv[argc++] = "-j";
	argv[argc++] = "-rightbar";
	argv[argc++] = "-s";
	argv[argc++] = "-sb";
	argv[argc++] = "-u8";
	argv[argc++] = "-wf";
	argv[argc++] = "-sl";
	argv[argc++] = "1024";
    }
    argv[argc++] = "-e";
    argv[argc++] = "/bin/sh";
    argv[argc++] = "-c";

    /* checks */
    {
       gchar *test,*prog1,*prog2;
       if(tree_details->preferences & RSH_X_SSH) prog1="rsh";
       else prog1="ssh";
       if(!(tree_details->preferences & RSYNC_X_SCP))prog2="rsync";
       else {
	       if(tree_details->preferences & RSH_X_SSH) prog2="rcp";
	       else prog2="scp";
       }
       test=g_find_program_in_path(prog1);
       if (!test) {
           print_diagnostics(treeview, "xf_ERROR_ICON",
			  strerror(ENOENT),":", prog1,"\n", NULL);
	   return FALSE;
       } else g_free(test);
       test=g_find_program_in_path(prog2);
       if (!test) {
           print_diagnostics(treeview, "xf_ERROR_ICON",
			  strerror(ENOENT),":", prog2,"\n", NULL);
	   return FALSE;
       } else g_free(test);
    }
       
	       
    if(!(tree_details->preferences & RSYNC_X_SCP))
    {
	if(tree_details->preferences & RSH_X_SSH)
	{
	    c = "rsync -av --rsh=rsh";
	}
	else
	{
	    c = "rsync -av --rsh=ssh";
	}
    }
    else
    {
	if(tree_details->preferences & RSH_X_SSH)
	{
	    c = "rcp -pr";
	}
	else
	    c = "scp -pr";
    }

    cursor_wait(treeview);
    pid = fork();
    if(pid)
    {
	do
	{
	    while(gtk_events_pending())
		gtk_main_iteration();
	}
	while(!waitpid(pid, &status, WNOHANG));
	cursor_reset(treeview);
	return TRUE;
    }
    
    /* child process continues here... */
    for(j = 0, length = 0; srcs[j]; j++)
    {
	/*printf("DBG:src=%s\n",srcs[j]); */
	length += (strlen(src_host) + 1 + strlen(srcs[j]) + 1);
    }
    all_srcs = (char *)malloc(length + 1);
    if(!all_srcs)
	g_assert_not_reached();
    all_srcs[0] = 0;
    for(j = 0; srcs[j]; j++)
    {
	strcat(all_srcs, src_host);
	strcat(all_srcs, ":");
	strcat(all_srcs, srcs[j]);
	strcat(all_srcs, " ");
    }

    cmd = (char *)malloc(strlen("echo ;") + 1 + 2 * strlen(c) + 2 + 2 * strlen(tgt) + 2 + 2 * length + 2);

    if(!cmd)
	g_assert_not_reached();
    sprintf(cmd, "echo %s %s %s;%s %s %s", c, all_srcs, tgt, c, all_srcs, tgt);
    /*printf("DBG:cmd=%s\n",cmd);*/
    argv[argc++] = cmd;

    argv[argc++] = 0;

    /*{int i;printf("DBG:");for(i=0;argv[i];i++)printf("%s ",argv[i]);printf("\n");} */


    execvp(argv[0], argv);
    _exit(123);
}


gboolean on_same_device(void)
{
    return same_device;
}

void set_override(gboolean state)
{
    force_override = state;
    return;
}

char *randomTmpName(char *ext)
{
    static char *fname = NULL;
    int fnamelen;
    if(fname)
	g_free(fname);
    if(ext == NULL)
	fnamelen = strlen("/tmp/xffm.XXXXXX") + 1;
    else
	fnamelen = strlen("/tmp/xffm.XXXXXX") + strlen(ext) + 2;
    fname = (char *)malloc(sizeof(char) * (fnamelen));
    if(!fname)
	return NULL;
    sprintf(fname, "/tmp/xffm.XXXXXX");
    close(mkstemp(fname));
    if(ext)
    {
	unlink(fname);
	strcat(fname, ".");
	strcat(fname, ext);
    }
    return fname;
}


char *SimpleTmpList(char *tgt, char *src)
{
    static char *fname = NULL;
    FILE *tmpfile;
    if((fname = randomTmpName(NULL)) == NULL)
	return NULL;
    if((tmpfile = fopen(fname, "w")) == NULL)
	return NULL;
    fprintf(tmpfile, "%d\t%s\t%s\n", TR_COPY, src, tgt);
    fclose(tmpfile);
    return fname;
}


char *mktgpath(char *target_dir, char *source)
{
    static char *target = NULL;
    char *name;
    name = strrchr(source, '/');
    if(!name)
	g_assert_not_reached();
    if(target)
	g_free(target);
    target = (char *)malloc(strlen(target_dir) + strlen(name) + 1);
    sprintf(target, "%s%s", target_dir, name);
    if(!target)
	g_assert_not_reached();
    return target;
}

static void warn_target_exists(GtkTreeView * treeview, char *target)
{
    /*tree_details_t *tree_details =get_tree_details(treeview); */
    if(force_override)
    {
	if(!waste) 
	   print_diagnostics(treeview, "xf_WARNING_ICON", 
			_("Overwrite"), " : ", target, "\n", NULL);
    }
    else
    {
	gchar *g=g_path_get_basename ((const gchar *)target);	
	dlg_result = DLG_CANCEL;
	make_dialog(treeview, NULL, g);
	g_free(g);
    }
}

/* function to check input before
 * packing off to copy/move/link
 * */
static int ok_input(GtkTreeView * treeview, char *target, tree_entry_t * s_en)
{
    struct stat t_stat;
    char *source;
    gboolean target_exists = TRUE;
    /*tree_details_t *tree_details =get_tree_details(treeview); */

    dlg_result = DLG_YES;
    source = s_en->path;
    if(lstat(target, &t_stat) < 0)
    {
	if(errno != ENOENT)
	{
	    print_diagnostics(treeview, "xf_ERROR_ICON", strerror(errno), " : ", source, "\n", NULL);
	    dlg_result = DLG_CONTINUE;
	    return dlg_result;
	}
	else
	    target_exists = FALSE;
    }
    /*fprintf(stderr,"dbg:at okinput %s->%s\n",s_en->path,target); */
    /* check for valid source */
    if(strstr(s_en->path, "tar:"))
	tar_extraction = TRUE;


    /*fprintf(stderr,"dbg:%s->%s\n",source,target); */

    /*fprintf(stderr,"dbg:at okinput 2 \n"); */
    /* target and source are the same */
    if((target_exists) && (t_stat.st_ino == s_en->st->st_ino))
    {
	/*fprintf(stderr,"dbg:nonsense input\n"); */
	print_diagnostics(treeview, "xf_ERROR_ICON", _("Source and target are the same!"), " (", source, ")\n", NULL);

	dlg_result = DLG_CANCEL_QUIET;
	return dlg_result;
    }

    if(!tar_extraction)
    {
	if(IS_XF_FIFO(s_en->type) || IS_XF_CHR(s_en->type) || IS_XF_BLK(s_en->type) || IS_XF_SOCK(s_en->type))
	{
	    print_diagnostics(treeview, "xf_ERROR_ICON", NULL);
	    if(IS_XF_FIFO(s_en->type))
		print_diagnostics(treeview, NULL, _("Can't copy FIFO"), NULL);
	    if(IS_XF_CHR(s_en->type))
		print_diagnostics(treeview, NULL, _("Can't copy caracter device"), NULL);
	    if(IS_XF_BLK(s_en->type))
		print_diagnostics(treeview, NULL, _("Can't copy block file"), NULL);
	    if(IS_XF_SOCK(s_en->type))
		print_diagnostics(treeview, NULL, _("Can't copy socket"), NULL);
	    print_diagnostics(treeview, NULL, " : ", source, "\n", NULL);
	    dlg_result = DLG_CONTINUE;
	    return dlg_result;
	}
    }

    /*fprintf(stderr,"dbg:at okinput 3 \n"); */
    /* check for overwrite, both files and directories */
  try_again:
    if(target_exists)
    {
	warn_target_exists(treeview, target);
	if(dlg_result == DLG_YES)
	{
	    if(waste)
	    {
		if(!wasteit(treeview, target))
		{
		    print_diagnostics(treeview, "xf_ERROR_ICON", 
				    _("Cannot move to wastebasket"), " ", 
				    target, " : ", strerror(errno), "\n", 
				    NULL);
		    print_status(treeview, "xf_ERROR_ICON", 
				    _("Cannot move to wastebasket"), 
				    NULL);

		    goto try_again;
		}
	    }
	    /* unlink case superceeded by overwrite */
	}


    }

    return dlg_result;
}


char *CreateTmpList(GtkTreeView * treeview, GList * list, tree_entry_t * t_en)
{
    char *target;
    FILE *tmpfile;
    uri *u;
    static char *fname = NULL;
    tree_entry_t *s_en;
    /*tree_details_t *tree_details =get_tree_details(treeview); */


    nitems = 0;
    if((fname = randomTmpName(NULL)) == NULL)
	return NULL;
    if((tmpfile = fopen(fname, "w")) == NULL)
	return NULL;
    same_device = TRUE;
    tar_extraction = FALSE;
    force_override = FALSE;
    for(; list != NULL; list = list->next)
    {
	int type = 0;
	struct stat t_stat;
	SET_LOCAL_TYPE(type);
	u = list->data;
	s_en = stat_entry(u->url, type);
	/* entry new has stated path by now */
	if(!s_en)
	{
	    /*fprintf(stderr,"dbg:s_en is NULL\n"); */
	    continue;
	}

	target = mktgpath(t_en->path, s_en->path);
	if (strcmp(t_en->path, s_en->path)==0){
		nitems = 0;		
#ifdef DEBUG
		printf("DBG:Cannot copy object upon itself\n");
#endif
		print_diagnostics(treeview,"xf_ERROR_ICON",
			_("Cannot copy object upon itself"),
			": ",target,"\n",NULL);
		destroy_entry(s_en);
		fclose(tmpfile);
		unlink(fname);
		return NULL;
	}
		

	/*printf("DBG: target=%s\n",t_en->path); */

	/* device determination */
	if(stat(target, &t_stat) < 0)
	{			/* follow link stat() */
	    char *route, *end;
	    route = (char *)malloc(strlen(target) + 1);
	    if(route)
	    {
		strcpy(route, target);
		end = strrchr(route, '/');
		if(end)
		{
		    if(end == route)
			end[1] = 0;	/* root directory */
		    else
			end[0] = 0;
		    if(stat(route, &t_stat) < 0)
		    {
			t_stat.st_dev = 113;
		    }
		}
		g_free(route);
	    }

	}

#ifdef DEBUG
	printf("DBG:target=%s\n",target); 
#endif
	switch (ok_input(treeview, target, s_en))
	{
	    case DLG_CONTINUE:
	    case DLG_NO:
		print_diagnostics(treeview, "xf_WARNING_ICON", _("Skipping"), " : ", s_en->path, "\n", NULL);
		break;
	    case DLG_CANCEL:	/* dnd cancelled */
		print_diagnostics(treeview, "xf_WARNING_ICON", _("Cancelled"), " : ", s_en->path, "\n", NULL);
	    case DLG_CANCEL_QUIET:	/* AUTOMATIC */
		destroy_entry(s_en);
		fclose(tmpfile);
		unlink(fname);
		return NULL;
	    case DLG_YES:
	    default:
		/*if ((u->type != URI_LOCAL)||(u->type != URI_FILE)) 
		 *  same_device=FALSE; else */
		if(s_en->st->st_dev != t_stat.st_dev)
		    same_device = FALSE;
		nitems++;
		fprintf(tmpfile, "%d\t%s\t%s\n", u->type, s_en->path, target);
		/*fprintf(stderr,"dbg:%d\t%s\t%s\n",u->type,s_en->path,target); */
		break;
	}
	destroy_entry(s_en);
    }
    fclose(tmpfile);
    if(!nitems)
    {
	unlink(fname);
	return NULL;
    }
    return fname;
}

int error_continue(GtkTreeView * treeview, char *err, char *file)
{
    /*tree_details_t *tree_details =get_tree_details(treeview); */
    if(force_override)
    {
	print_diagnostics(treeview, "xf_ERROR_ICON", err, " : ", file, "\n", NULL);
	dlg_result = DLG_CANCEL;
    }
    else
    {
	dlg_result = DLG_CANCEL;
	make_dialog(treeview, err, file);
    }
    return dlg_result;
}


/* function for non forked move on same device, or symlink too 
 * since this is not recursive, file overwrite has already
 * been checked with user.*/
gboolean DirectTransfer(GtkTreeView * treeview, int mode, char *tmpfile)
{
    FILE *tfile;
    char line[MAX_LINE_SIZE], *source, *target;
    int type, result = 0;
    struct stat st;
    /*fprintf(stderr,"dbg: at DirectTransfer\n"); */

    tfile = fopen(tmpfile, "r");
    if(!tfile)
    {
	print_diagnostics(treeview, "xf_ERROR_ICON", strerror(errno), tmpfile, "\n", NULL);
	return FALSE;
    }
    while(!feof(tfile) && fgets(line, MAX_LINE_SIZE - 1, tfile))
    {
	line[MAX_LINE_SIZE - 1] = 0;
	type = atoi(strtok(line, "\t"));
	source = strtok(NULL, "\n");
	target = strrchr(source, '\t') + 1;
	*(strrchr(source, '\t')) = 0;
      try_again:
	/* this is queried in tmpfile creation */
	if(lstat(target, &st) >= 0)
	    unlink(target);
	lstat(source, &st);
	if(S_ISLNK(st.st_mode))
	{
	    char *src_lnk;
	    int len;
	    struct stat st;
	    lstat(source, &st);
	    src_lnk = (char *)malloc(st.st_size + 1);
	    if(!src_lnk)
		continue;
	    len = readlink(source, src_lnk, st.st_size);
	    if(len <= 0)
	    {
		g_free(src_lnk);
		switch (error_continue(treeview, strerror(errno), source))
		{
		    case DLG_CANCEL:
			goto transfer_over;
		    case DLG_YES:
			goto try_again;
		    case DLG_NO:
		    default:
			continue;
		}
	    }
	    src_lnk[len] = 0;
	    result = symlink(src_lnk, target);
	    g_free(src_lnk);
	    if(result >= 0 && (mode & TR_MOVE) && (unlink(source) < 0))
	    {
		print_diagnostics(treeview, "xf_ERROR_ICON", strerror(errno), " : ", source, "\n", NULL);
	    }
	    continue;
	}
#if 0
	if(tar_extraction)
	{
	    char *o_src;
	    o_src = g_strdup(source);
	    /* strtoks down the line... */
	    if(strncmp(source, "tar:", strlen("tar:")) != 0)
		continue;
	    tar_extract((GtkCTree *) ctree, target, source);
	    if(mode == TR_MOVE)
		tar_delete((GtkCTree *) ctree, o_src);
	    g_free(o_src);
	}
#endif
	else if(mode & TR_MOVE)
	    result = rename(source, target);
	else if(mode & TR_LINK)
	    result = symlink(source, target);
	else
	    g_assert_not_reached();

	if(result < 0)
	{
	    switch (error_continue(treeview, strerror(errno), target))
	    {
		case DLG_CANCEL:
		    goto transfer_over;
		case DLG_YES:
		    goto try_again;
		case DLG_NO:
		default:
		    continue;

	    }
	}
	/* here ends the read from tmpfile loop */
    }
  transfer_over:
    fclose(tfile);
    return TRUE;
}


/* the recursive_count_files requieres a lstat (for compatibility) since
 * d_type is not quite standard.
 *  */
int recursive_count_files(char *path)
{
    DIR *directory;
    struct dirent *dir;
    char fullpath[_POSIX_PATH_MAX];
    struct stat st;

    directory = opendir(path);
    if(!directory)
    {
	return -1;
    }
    while((dir = readdir(directory)) != NULL && !count_cancelled)
    {
	if(strcmp(dir->d_name, ".") == 0 || strcmp(dir->d_name, "..") == 0)
	    continue;
	if(strlen(path) + strlen(dir->d_name) + 2 > _POSIX_PATH_MAX)
	{
	    printf("DBG:%s/%s\n", path, dir->d_name);
	    g_assert_not_reached();
	}
	sprintf(fullpath, "%s/%s", path, dir->d_name);
	lstat(fullpath, &st);
	countT++;
	smallcount++;
	sizeT += st.st_size;
	if(S_ISDIR(st.st_mode))
	    recursive_count_files(fullpath);
	else if(smallcount & 0x100)
	{
	    smallcount = 0;
	    sprintf(fullpath, "%d %s", countT, _("files"));
	    gtk_label_set_text((GtkLabel *) count_label, fullpath);
    	    process_pending_gtk();
	}
    }
    closedir(directory);
    return (1);
}

static gint ParentCount(gpointer data)
{
    FILE *tfile;
    char *line, *source;
    int type;
    GtkTreeView *treeview = (GtkTreeView *) data;
    struct stat st;

    g_source_remove(counttimer);
    counttimer = 0;
    smallcount = 0;
    countT = 0;
    sizeT = 0;
    line = (char *)malloc(MAX_LINE_SIZE);
    if(!line)
	g_assert_not_reached();
    tfile = fopen(child_file, "r");
    if(!tfile)
	g_assert_not_reached();
    while(fgets(line, MAX_LINE_SIZE - 1, tfile) && !feof(tfile) && !count_cancelled)
    {
	/*fprintf(stderr,"dbg:%s\n",line); */
	type = atoi(strtok(line, "\t"));
	source = strtok(NULL, "\t");
	recursive_count_files(source);
	lstat(source, &st);
	countT++;
	sizeT += st.st_size;
	sprintf(line, "%s %s", _("Counted"), sizetag(sizeT, countT));
	print_status(treeview, "xf_INFO_ICON", line, NULL);
    }
    fclose(tfile);
    g_free(line);


    gtk_main_quit();
    return FALSE;
}


static void cb_count_destroy(GtkWidget * widget, gpointer data)
{
    count_cancelled = TRUE;	/* for user destruction */
    countW = NULL;
}
static void cb_count_cancel(GtkWidget * widget, gpointer data)
{
    count_cancelled = TRUE;	/* for user destruction */
    gtk_widget_destroy(countW);
    countW = NULL;
}

static void count_window(GtkTreeView * treeview)
{
    GtkWidget *cancel;
    tree_details_t *tree_details = get_tree_details(treeview);
    count_cancelled = FALSE;
    countW = gtk_dialog_new();

    gtk_window_set_position(GTK_WINDOW(countW), GTK_WIN_POS_MOUSE);
    gtk_window_set_modal(GTK_WINDOW(countW), TRUE);
    count_label = gtk_label_new(_("Counting files"));
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(countW)->vbox), count_label, TRUE, TRUE, 3);

    count_label = gtk_label_new(".........................................");
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(countW)->vbox), count_label, TRUE, TRUE, 3);

    cancel = gtk_button_new_with_label(_("Cancel"));
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(countW)->action_area), cancel, FALSE, FALSE, 3);
    g_signal_connect(G_OBJECT(cancel), "clicked", (GCallback)cb_count_cancel, NULL);

    gtk_widget_realize(countW);

    /*if (preferences&SMALL_DIALOGS) 
       gdk_window_set_decorations (countW->window,GDK_DECOR_BORDER); */

    if(tree_details->window)
	gtk_window_set_transient_for(GTK_WINDOW(countW), GTK_WINDOW(tree_details->window));
    g_signal_connect(GTK_OBJECT(countW), "destroy", (GCallback)cb_count_destroy, NULL);
    gtk_widget_show_all(countW);
    gdk_flush();
    /* add timeout */
    counttimer = g_timeout_add_full(0, 260, (GtkFunction) ParentCount, (gpointer) treeview, NULL);
    /* deprecated: counttimer = gtk_timeout_add(260, (GtkFunction) ParentCount, (gpointer) treeview);*/


    gtk_main();
    if(countW)
    {
	gtk_widget_destroy(countW);
	count_cancelled = FALSE;
    }
    return;
}

static int rwStderr(int n, void *data);
static int rwStdout(int n, void *data);
static void rwForkOver(pid_t pid);
static void ChildTransfer(gpointer data);
static void set_innerloop(gboolean state);
static int internal_rw_file(char *target, char *source, off_t size);

gboolean IndirectTransfer(GtkTreeView * treeview, int mode, char *tmpfile)
{
    /*GtkWidget *cpy_dlg=NULL; */
    tree_details_t *tree_details = get_tree_details(treeview);

    cpy_treeview = treeview;
    if(CHILD_FILE_LENGTH < strlen("/tmp/xffm.XXXXXX") + 1)
	g_assert_not_reached();
    strncpy(child_file, tmpfile, CHILD_FILE_LENGTH);
    child_file[CHILD_FILE_LENGTH - 1] = (char)0;

    child_mode = mode;

    /* count total files to transfer */
    count_window(treeview);
    if(count_cancelled)
	return TRUE;

    /*fprintf(stderr,"dbg:Total files=%d\n",countT); */

    initial = time(NULL);
    show_stop(tree_details->window);
    rw_fork_obj = Tubo(ChildTransfer, NULL, rwForkOver, TRUE, rwStdout, rwStderr);

    /* only parent continues from here */

    /*fprintf(stderr,"dbg:call to innerloop from IndirectTransfer()\n"); */
    set_innerloop(TRUE);

    set_progress(treeview, countT, countT);

    return TRUE;
}


/* callbacks */


void on_countfiles_activate(GtkMenuItem * menuitem, gpointer user_data)
{
    GtkTreeView *treeview = get_selected_treeview((GtkWidget *) menuitem);
    GtkTreeSelection *selection = gtk_tree_view_get_selection(treeview);
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    /*tree_details_t *tree_details=get_tree_details(treeview); */
    tree_entry_t *en;
    GtkTreeIter iter;

    gtk_tree_selection_set_mode(selection, GTK_SELECTION_BROWSE);
    if(gtk_tree_selection_get_selected(selection, &treemodel, &iter))
    {
	gtk_tree_model_get(treemodel, &iter, ENTRY_COLUMN, &en, -1);
	strcpy(child_file, SimpleTmpList(en->path, en->path));
	count_window(treeview);
	unlink(child_file);
    }
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);
    /* count total files to transfer */
    return;
}

/*  copy stuff */

static gboolean SubChildTransfer(char *target, char *source)
{
    struct stat s_stat, t_stat;
    int rc;

    if(stat(target, &t_stat) < 0)
    {				/* follow link for target: stat() */
	char *route, *end;
	route = (char *)malloc(strlen(target) + 1);
	if(route)
	{
	    strcpy(route, target);
	    if((end = strrchr(route, '/')) != NULL)
	    {
		if(end == route)
		    end[1] = 0;	/* root directory */
		else
		    end[0] = 0;
		stat(route, &t_stat);
	    }
	    g_free(route);
	}
    }
    /* do the link business before going recursive:
     * (link directories, not contents) */
    /* (historical, TR_LINK should fall to DirectTransfer) */
    if(child_mode & TR_LINK)
	g_assert_not_reached();

    lstat(source, &s_stat);	/* stat() the link itself */

/* what if we find a symlink in our path? */
    if(S_ISLNK(s_stat.st_mode))
    {
	char *src_lnk;
	int len;
	struct stat st;
	lstat(source, &st);
	src_lnk = (char *)malloc(st.st_size + 1);
	if(!src_lnk)
	    return TRUE;
	len = readlink(source, src_lnk, st.st_size);
	if(len <= 0)
	{
	    fprintf(stderr, "xffm:%s\n", strerror(errno));
	    return (TRUE);
	}
	src_lnk[len] = 0;
	if(lstat(target, &st) == 0)
	{
	    if(unlink(target) < 0)
		return process_error("rm", RW_ERROR_WRITING_TGT);
	}

	if(symlink(src_lnk, target) < 0)
	{
	    g_free(src_lnk);
	    return process_error("ln", RW_ERROR_WRITING_TGT);
	}
	g_free(src_lnk);
	fprintf(stdout, "child:item:%d\n", ++child_path_number);
	if(child_mode & TR_MOVE)
	{
	    if(unlink(source) < 0)
	    {
		return process_error("rm", RW_ERROR_WRITING_SRC);
	    }
	}
	return (TRUE);
    }

    /* recursivity, if src is directory:
     * 1- mkdir tgt/src
     * 2- glob src
     * 3 foreach glob in globlist recall function.
     * */
    /* non GNU fallback: wont copy hidden files in recursivity
     * unless specifically selected */

    if(S_ISDIR(s_stat.st_mode))
    {				/* directories fall in here, going recursive */
	DIR *directory;
	struct dirent *dir;
	char newtarget[_POSIX_PATH_MAX], fullpath[_POSIX_PATH_MAX];

	/* create target dir */
	if(mkdir(target, (s_stat.st_mode | 0700)) < 0)
	{
	    /* dont process error if directory exists */
	    if(errno != EEXIST)
	    {
		targetdir = target;
		process_error("mkdir", RW_ERROR_WRITING_DIR);
	    }
	}
	fprintf(stdout, "child:item:%d\n", ++child_path_number);
	fprintf(stdout, "child:bytes:%lld\n", (long long)((long long unsigned)sizeT / 1024));
	fflush(NULL);

	/*fprintf(stderr,"dbg:dir created: %s\n",target); */

	directory = opendir(source);
	if(!directory)
	{
	    return -1;
	}
	while((dir = readdir(directory)) != NULL)
	{
	    if(strcmp(dir->d_name, ".") == 0 || strcmp(dir->d_name, "..") == 0)
		continue;
	    if(strlen(source) + strlen(dir->d_name) + 2 > _POSIX_PATH_MAX)
	    {
		printf("DBG:%s/%s\n", source, dir->d_name);
		g_assert_not_reached();
	    }
	    sprintf(fullpath, "%s/%s", source, dir->d_name);
	    sprintf(newtarget, "%s/%s", target, dir->d_name);

	    if(!SubChildTransfer(newtarget, fullpath))
	    {
		closedir(directory);
		return FALSE;
	    }
	}
	closedir(directory);
	/* remove old directory (rmdir should fail if any interior move failed) */
	if((child_mode & TR_MOVE) && (rmdir(source) < 0))
	{
	    process_error("rmdir", RW_ERROR_WRITING_TGT);
	    /*return process_error(RW_ERROR_WRITING_TGT); */
	    /*return FALSE; */
	}
	return TRUE;
    }

    if((child_mode & TR_MOVE) && (s_stat.st_dev == t_stat.st_dev))
    {
	if(rename(source, target) < 0)
	{
	    return process_error("mv", RW_ERROR_WRITING_TGT);
	}
	fprintf(stdout, "child:item:%d\n", ++child_path_number);
	fprintf(stdout, "child:bytes:%lld\n", (long long)((long long unsigned)sizeT / 1024));
	fflush(NULL);
	return TRUE;
    }


    if(S_ISFIFO(s_stat.st_mode))
    {
	return process_error("cp", RW_ERROR_FIFO);
    }
    if(S_ISCHR(s_stat.st_mode) || S_ISBLK(s_stat.st_mode))
    {
	return process_error("cp", RW_ERROR_DEVICE);
    }
    if(S_ISSOCK(s_stat.st_mode))
    {
	return process_error("cp", RW_ERROR_SOCKET);
    }
    /* we have to copy the data by reading/writing  */
    rc = internal_rw_file(target, source, s_stat.st_size);
    if (!rc) return rc;

    if(!(rc & (RW_ERROR_WRITING_TGT | RW_ERROR_OPENING_TGT)))
    {
	/* put target with original file mode: */
#if DEBUG
	  printf("DBG:doing chmod(%s)...",target);fflush(NULL);
#endif
 	if(chmod(target, s_stat.st_mode) < 0)
	{
	    return process_error("chmod", RW_ERROR_STAT_WRITE_TGT);
	}
#if DEBUG
	  printf("ok\n",target);
#endif
 	
/*	neither copied nor moved files should have modification
 *	date altered (same file contents, man!) */
	if (TRUE) {
  	  struct utimbuf times;
  	  times.modtime=s_stat.st_mtime; 
  	  times.actime=s_stat.st_atime;   
	  /* set the file time stuff to original file time */
#if DEBUG
	  printf("DBG:doing utime(%s)...",target);fflush(NULL);
#endif
  	  if(utime(target,&times)< 0){	
	    return process_error("utime", RW_ERROR_WRITING_SRC);
	  }	  
#if DEBUG
	  printf("ok\n",target);
#endif
	}
	
	if (child_mode & TR_MOVE) {
	  /* if all worked, try removing original file */
#if DEBUG
	  printf("DBG:doing unlink(%s)...",source);fflush(NULL);
#endif
	  if(unlink(source) < 0){
	    return process_error("rm", RW_ERROR_WRITING_SRC);
	  }
#if DEBUG
	  printf("ok\n",target);
#endif
	}

    }
    /* wow. everything went ok. */
    return TRUE;
}

void finish(int sig)
{
    if(incomplete_target)
	unlink(fork_target);
    unlink(child_file);
    _exit(123);
}

static void ChildTransfer(gpointer data)
{
    FILE *tfile;
    char *line, *source, *target;
    int type;

    all_recursive = FALSE;
    I_am_child = TRUE;
    {
	struct sigaction act;
	act.sa_handler = finish;
	sigemptyset(&act.sa_mask);
#ifdef SA_RESTART
	act.sa_flags = SA_RESTART;
#else
	act.sa_flags = 0;
#endif
	sigaction(SIGTERM,&act,NULL);
    }
    incomplete_target = FALSE;
    line = (char *)malloc(MAX_LINE_SIZE);
    if(!line)
    {				/* fatal error */
	g_assert_not_reached();
    }
    tfile = fopen(child_file, "r");
    if(!tfile)
    {				/* fatal error */
	process_error("open", RW_ERRNO);
	unlink(holdfile);
	_exit(123);
    }
    child_path_number = 0;
    sizeT = 0;
    while(!feof(tfile) && fgets(line, MAX_LINE_SIZE - 1, tfile))
    {
	line[MAX_LINE_SIZE - 1] = 0;
	type = atoi(strtok(line, "\t"));
	source = strtok(NULL, "\n");
	target = strrchr(source, '\t') + 1;
	*(strrchr(source, '\t')) = 0;
	switch (type)
	{
	    case URI_LOCAL:
	    case URI_FILE:
		/*source=strtok(NULL,"\t");target=strtok(NULL,"\n"); */
		if(!SubChildTransfer(target, source))
		    goto cut_out;
		break;
	    case URI_HTTP:
	    case URI_FTP:
		fprintf(stderr, "not implemented (%d) %s->%s\n", type, source, target);
		/*download(target,source); */
		break;
	    case URI_TAR:
		break;
	    default:
		/*fprintf(stdout,"child:unknown type (%d) %s\n",type,source); */
		fprintf(stderr, "unknown type (%d) %s->%s\n", type, source, target);
		break;
	}
    }
  cut_out:
    fclose(tfile);
    g_free(line);
    fflush(NULL);
    _exit(123);
}

/* 
 * recursive function for directories
 * */
#define BUFFER_SIZE (4096)
static int internal_rw_file(char *target, char *source, off_t size)
{
    int i, j = 0, source_file, target_file;
    char *buffer;
    off_t total_size = 0;
    gboolean too_few = FALSE, too_many = FALSE;
    int b_size = BUFFER_SIZE;
    struct stat statbuf;


    fork_target = target;
    fork_source = source;
    /* open source */
    source_file = open(source, O_RDONLY);
    if(source_file < 0)
    {
	process_error("open", RW_ERROR_OPENING_SRC);
	return (RW_ERROR_OPENING_SRC);
    }
    /* open target */
    target_file = open(target, O_WRONLY | O_TRUNC | O_CREAT);
    if(target_file < 0)
    {
	close(source_file);
	process_error("open", RW_ERROR_OPENING_TGT);
	return (RW_ERROR_OPENING_TGT);
    }

    /* assign optimum block size */
    if(fstat(source_file, &statbuf) == 0)
	b_size = ST_BLKSIZE(statbuf);
    /* allocate buffer */
    buffer = (char *)malloc(BUFFER_SIZE);
    if(!buffer)
	g_assert_not_reached();

    incomplete_target = TRUE;
    /* read/write loop */
    /* 
     * interruption occurs with signal SIGTERM
     * sent from parent process via the dialog
     * in which case parent proceeds with cpyForkCleanup()
     * Closing the source file is not necesary 
     * because the child process will terminate, closing it then. 
     * */
    /*fprintf(stdout,"child:bytes: %d / %d \n",child_file_number-1,countT);fflush(NULL); */
    fprintf(stdout, "child:item: %d \n", ++child_path_number);
    fprintf(stdout, "child:bytes:%lld\n", (long long)((long long unsigned)sizeT / 1024));
    fflush(NULL);
    while((i = read(source_file, buffer, BUFFER_SIZE)) > 0)
    {
	if((j = write(target_file, buffer, i)) < 0)
	    break;
	if(i > j)
	{
	    too_few = TRUE;
	    break;
	}
	if(i < j)
	{
	    too_many = TRUE;
	    break;
	}
	total_size += j;
	sizeT += j;
	smallcount++;
	if(smallcount & 0x100)
	{
	    smallcount = 0;
	    fprintf(stdout, "child:bytes:%lld\n", (long long)((long long unsigned)sizeT / 1024));
	    fflush(NULL);
	    usleep(100);	/* only with large files*/
	}
    }
    g_free(buffer);
    if(close(source_file) < 0)
    {
	close(target_file);
	process_error("close", RW_ERROR_CLOSE_SRC);
	return (RW_ERROR_CLOSE_SRC);
    }
    if(close(target_file) < 0)
    {
	process_error("close", RW_ERROR_CLOSE_TGT);
	return (RW_ERROR_CLOSE_TGT);
    }
    incomplete_target = FALSE;

    if((i < 0) || (j < 0) || (too_few) || (too_many))
    {
	if(unlink(target))
	{
	    process_error("rm", RW_ERROR_UNLINK_SRC);
	    return (RW_ERROR_UNLINK_SRC);
	}
	if(too_few)
	{
	    process_error("cp", RW_ERROR_TOO_FEW);
	    return (RW_ERROR_TOO_FEW);
	}
	if(too_many)
	{
	    process_error("cp", RW_ERROR_TOO_MANY);
	    return (RW_ERROR_TOO_MANY);
	}
	if(i < 0)
	{
	    process_error("cp", RW_ERROR_READING_SRC);
	    return (RW_ERROR_READING_SRC);
	}
	if(j < 0)
	{
	    process_error("cp", RW_ERROR_WRITING_TGT);
	    return (RW_ERROR_WRITING_TGT);
	}
    }

    if(total_size < size)
    {
	process_error("cp", RW_ERROR_TOO_FEW);
	return (RW_ERROR_TOO_FEW);
    }
    /* everything went OK */
    return (RW_OK);
}

/* FUNCTION start/stop inner gtk_main() */
static void set_innerloop(gboolean state)
{
    static gboolean innerloop = FALSE;
    if(state == innerloop)
	return;
    innerloop = state;
    /*fprintf(stderr,"dbg:innerloop %s\n",(state)?"started":"terminated");fflush(NULL); */
    if(state)
	gtk_main();
    else
	gtk_main_quit();
}

/* function to process stderr produced by child */
static int rwStderr(int n, void *data)
{
    char *line;

    if(n)
	return TRUE;		/* this would mean binary data */
    line = (char *)data;
    fprintf(stderr, "child stderr:%s\n", line);
    return TRUE;
}

/* function to clean up the mess after child process
 * has been sent a SIGTERM. (Might a unlink target be 
 * good here?)
 * */
static void cpyForkCleanup(void)
{
    /* break inner gtk_main() loop */
    /*fprintf(stderr,"dbg:call to innerloop from forkcleanup\n"); */
    set_innerloop(FALSE);
    /* unlink(fork_target); */
}

/* function to process stdout produced by child */
static int rwStdout(int n, void *data)
{
    char *line, *texto;
    tree_details_t *tree_details = get_tree_details(cpy_treeview);

    if(n)
	return TRUE;		/* this would mean binary data */
    line = (char *)data;

    if(stop)
    {
	stop = FALSE;
	hide_stop(tree_details->window);
	rw_fork_obj = TuboCancel(rw_fork_obj, cpyForkCleanup);
	print_status(cpy_treeview, "xf_ERROR_ICON", 
		strerror(ETIMEDOUT),
			NULL);
	return TRUE;
    }


    /*printf("dbg(rwStdout):%s",line);fflush(NULL); */

    /* only process child specific lines: */
    if(strncmp(line, "child:", strlen("child:")) != 0)
    {
	return TRUE;
    }
/* clear to process next file */
    if(strncmp(line, "child:runOver", strlen("child:runOver")) == 0)
    {
	set_innerloop(FALSE);
	return TRUE;
    }


    /* src-tgt update */
    if(strncmp(line, "child:tgt-src:", strlen("child:tgt-src:")) == 0)
    {
	char *src, *tgt;
	strtok(line, ":");
	strtok(NULL, ":");
	tgt = strtok(NULL, "\t");
	src = strtok(NULL, "\n");
	if(tgt && src && strlen(src) >= 2)
	{
	    print_status(cpy_treeview, "xf_INFO_ICON", _("Copying"), " ", __VY_FN__(src), NULL);
	}
	return TRUE;
    }
    /* nitems update */
    if(strncmp(line, "child:item:", strlen("child:item:")) == 0)
    {
	int n;
	strtok(line, ":");
	strtok(NULL, ":");
	n = atoi(strtok(NULL, "\n"));
	/*printf("DBG: item %d/%d\n",n,countT); */
#ifdef STATUS_BY_ITEMS
	set_progress(cpy_treeview, n, countT);
#else
	nitems = n;
#endif
	return TRUE;
    }
    /* bytes update */
    if(strncmp(line, "child:bytes:", strlen("child:bytes:")) == 0)
    {
	long n;
	strtok(line, ":");
	strtok(NULL, ":");
	n = atol(strtok(NULL, "\n"));
	/*printf("DBG: bytes %ld/%lld\n",n,(long long)((long long unsigned)sizeT/1024)); */
	set_progress_bytes(cpy_treeview, n, (long long)((long long unsigned)sizeT / 1024));
	return TRUE;
    }

    /* anything else is a process_error() and operation
     * should continue with a diagnostics message.  */
    strtok(line, ":");
    texto = strtok(NULL, "\n");
    /*printf("dbg(rwStdout error):%s\n",texto);fflush(NULL); */

    print_diagnostics(cpy_treeview, "xf_ERROR_ICON", texto, "\n", NULL);
    process_pending_gtk();


    return TRUE;
}

/* function called when child is dead */
static void rwForkOver(pid_t pid)
{
#if 0
    char *allfile;
    allfile = (char *)malloc(strlen(child_file) + 1 + strlen(".all"));
    if(allfile)
    {
	sprintf(allfile, "%s.all", child_file);
	unlink(allfile);
	g_free(allfile);
    }
#endif
    char secs[1024];
    tree_details_t *tree_details = get_tree_details(cpy_treeview);
    rw_fork_obj = NULL;
    /*fprintf(stderr,"dbg: call to innerloop from forkover()\n");fflush(NULL); */
    set_innerloop(FALSE);
    set_progress(cpy_treeview, -1, -1);
    sprintf(secs, "%d/%d %s %ld %s", nitems, countT, _("files copied in"), (long)(time(NULL) - initial), (time(NULL) - initial == 1) ? _("second") : _("seconds"));
    print_status(cpy_treeview, "xf_INFO_ICON", my_cut_utf_string(secs), NULL);
    hide_stop(tree_details->window);
    stop = FALSE;
}
