/*  xfce4
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifndef PATH_MAX
#define PATH_MAX 1024
#endif

#include <libxfce4util/libxfce4util.h>

#include "icons.h"

#if GTK_CHECK_VERSION(2, 4, 0)

#include <gtk/gtkicontheme.h>

static GtkIconTheme *icon_theme = NULL;

#else

#include <glib.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#define DEFAULT_ICON_THEME "Curve"
static gchar icon_theme[256] = DEFAULT_ICON_THEME;

/* nb: when adding paths here, make sure they have a trailing '/' */
static const gchar *pix_paths[] = {
	DATADIR "/xfce4/icons/%s/scalable/emblems/",  /* for xfce4 theme-specific path */
	DATADIR "/xfce4/icons/%s/48x48/emblems/",
	DATADIR "/xfce4/icons/%s/24x24/emblems/",
	DATADIR "/xfce4/icons/%s/22x22/emblems/",
	DATADIR "/icons/%s/scalable/emblems/",  /* for xfce4 theme-specific path */
	DATADIR "/icons/%s/48x48/emblems/",
	DATADIR "/icons/%s/24x24/emblems/",
	DATADIR "/icons/%s/22x22/emblems/",
	"/usr/share/icons/%s/scalable/apps/",  /* ditto */
	"/usr/share/icons/%s/48x48/apps/",  /* ditto */
	"/usr/share/icons/%s/32x32/apps/",  /* ditto */
	"/usr/share/pixmaps/",
	"/usr/share/icons/hicolor/scalable/apps/",
	"/usr/share/icons/hicolor/48x48/apps/",
	"/usr/share/icons/hicolor/32x32/apps/",
	"/usr/share/icons/gnome/scalable/apps/",  /* gnome's default */
	"/usr/share/icons/gnome/48x48/apps/",  /* ditto */
	"/usr/share/icons/gnome/32x32/apps/",  /* ditto */
	"/usr/share/icons/default.kde/scalable/apps/",  /* kde's default */
	"/usr/share/icons/default.kde/48x48/apps/",  /* ditto */
	"/usr/share/icons/default.kde/32x32/apps/",  /* ditto */
	"/usr/share/icons/",  /* broken, but some apps do it anyway */
	NULL
};

static const gchar *user_paths[] = {
	"%s/.icons/%s",
	"%s/.icons/hicolor/scalable/apps/%s",
	"%s/.icons/hicolor/48x48/apps/%s",
	"%s/.icons/hicolor/32x32/apps/%s",
	"%s/.pixmaps/%s",
	NULL
};

static const gchar *kdefmts[] = {
	"%s/share/icons/default.kde/scalable/apps/%s",
	"%s/share/icons/default.kde/48x48/apps/%s",
	"%s/share/icons/default.kde/32x32/apps/%s",
	"%s/share/icons/hicolor/scalable/apps/%s",
	"%s/share/icons/hicolor/48x48/apps/%s",
	"%s/share/icons/hicolor/32x32/apps/%s",
	NULL
};

static const gchar *pix_ext[] = {
	".svg",
	".svgz",
	".png",
	".xpm",
	NULL
};

#endif

/*
 * create a GdkPixbuf from inline data and scale it to a given size
 */
GdkPixbuf *
xfce_inline_icon_at_size (const guint8 * data, int width, int height)
{

    GdkPixbuf *base;

    base = gdk_pixbuf_new_from_inline (-1, data, FALSE, NULL);

    g_assert (base);

    if ((width < 0 && height < 0)
	|| (gdk_pixbuf_get_width (base) == width
	    && gdk_pixbuf_get_height (base) == height))
    {
	return base;
    }
    else
    {
	GdkPixbuf *scaled;

	scaled =
	    gdk_pixbuf_scale_simple (base,
				     width >
				     0 ? width : gdk_pixbuf_get_width (base),
				     height >
				     0 ? height :
				     gdk_pixbuf_get_height (base),
				     GDK_INTERP_BILINEAR);

	g_object_unref (G_OBJECT (base));

	return scaled;
    }
}

/* deprecated */
GdkPixbuf *
inline_icon_at_size (const guint8 * data, int width, int height)
{
	return xfce_inline_icon_at_size(data, width, height);
}

#if GTK_CHECK_VERSION(2, 4, 0)
static void
ensure_icon_theme()
{
	const gchar *kdedir = g_getenv("KDEDIR");
	gchar *kde_icon_dir = NULL;
	gchar **paths;
	gint n;

	if(icon_theme)
		return;
	
	icon_theme = gtk_icon_theme_get_default();

	/* setup XDG icon paths */
	paths = xfce_resource_lookup_all(XFCE_RESOURCE_DATA, "icons/");
	for(n = 0; paths[n] != NULL; ++n)
		gtk_icon_theme_prepend_search_path(icon_theme, paths[n]);
	g_strfreev(paths);
	
	gtk_icon_theme_prepend_search_path(icon_theme, DATADIR "/xfce4/icons");
	gtk_icon_theme_prepend_search_path(icon_theme, DATADIR "/icons");
	gtk_icon_theme_prepend_search_path(icon_theme, DATADIR "/pixmaps");
	if(kdedir && *kdedir && strcmp(kdedir, "/usr") && strcmp(kdedir, PREFIX)) {
		kde_icon_dir = g_strconcat(kdedir, "/share/icons", NULL);
		gtk_icon_theme_append_search_path(icon_theme, kde_icon_dir);
		g_free(kde_icon_dir);
	}
}

#else /* if !GTK_CHECK_VERSION(2, 4, 0) */

static gboolean
icon_find_prefix(const gchar *prefix, const gchar **dirs, const gchar *filename,
		gchar *icon_path)
{
	gint i, j;
	
	if(!prefix || !dirs || !filename || !icon_path)
		return FALSE;
	
	for(i=0; dirs[i]; i++) {
		g_snprintf(icon_path, PATH_MAX, dirs[i], prefix, filename);
		if(g_strrstr(icon_path, ".") <= g_strrstr(icon_path, "/")) {
			int len = strlen(icon_path);
			for(j=0; pix_ext[j]; j++) {
				icon_path[len] = 0;
				g_strlcat(icon_path, pix_ext[j], PATH_MAX);
				if(g_file_test(icon_path, G_FILE_TEST_EXISTS))
					return TRUE;
			}
		} else {
			if(g_file_test(icon_path, G_FILE_TEST_EXISTS))
				return TRUE;
		}
	}
	
	return FALSE;
}

static gchar *
icon_lookup(const gchar *filename)
{
	gboolean found = FALSE;
	gint i, j;
	gchar icon_path[PATH_MAX];
	const gchar *kdedir = g_getenv("KDEDIR");
	
	g_return_val_if_fail(filename != NULL, NULL);
	
	if(*filename == '/') {
		g_strlcpy(icon_path, filename, PATH_MAX);
		if(g_file_test(icon_path, G_FILE_TEST_EXISTS))
			found = TRUE;
	} else {
		for(i=0; pix_paths[i] && !found; i++) {
			if(strstr(pix_paths[i], "%s")) {
				if(!icon_theme[0])
					continue;
				g_snprintf(icon_path, PATH_MAX, pix_paths[i], icon_theme);
			} else
				g_strlcpy(icon_path, pix_paths[i], PATH_MAX);
			g_strlcat(icon_path, filename, PATH_MAX);
			if(g_strrstr(icon_path, ".") <= g_strrstr(icon_path, "/")) {
				int len = strlen(icon_path);
				for(j=0; pix_ext[j] && !found; j++) {
					icon_path[len] = 0;
					g_strlcat(icon_path, pix_ext[j], PATH_MAX);
					if(g_file_test(icon_path, G_FILE_TEST_EXISTS))
						found = TRUE;
				}
			} else {
				if(g_file_test(icon_path, G_FILE_TEST_EXISTS))
					found = TRUE;
			}
		}
		if(!found && kdedir && *kdedir=='/' && strcmp(kdedir, "/usr"))
			found = icon_find_prefix(kdedir, kdefmts, filename, icon_path);
		if(!found)
			found = icon_find_prefix((const gchar *)xfce_get_homedir(),
					user_paths,	filename, icon_path);
	}
	
	if(found)
		return g_strdup(icon_path);
	
	return NULL;
}

static GdkPixbuf *
icon_find(const gchar *filename, gint size)
{
	gchar *icon_path = NULL;
	GdkPixbuf *miicon = NULL, *tmp;
	gint w, h;

	icon_path = icon_lookup(filename);
	
	if(icon_path) {
		miicon = gdk_pixbuf_new_from_file(icon_path, NULL);
		g_free(icon_path);
		
		if (miicon) {
			w = gdk_pixbuf_get_width(miicon);
			h = gdk_pixbuf_get_height(miicon);
			if (w != size || h != size) {
				GdkPixbuf *tmp;
				tmp = gdk_pixbuf_scale_simple(miicon, size, size, GDK_INTERP_BILINEAR);
				g_object_unref(G_OBJECT(miicon));
				miicon = tmp;
			}
		}
	}
    
    return miicon;
}

#endif /* if !GTK_CHECK_VERSION(2, 4, 0) */

/**
 * xfce_set_icon_theme:
 * @theme_name: The name of the icon theme.
 *
 * Sets the icon theme that xfce_load_themed_icon() will use when looking up
 * icons.  Note: this function amounts to a no-op if libxfcegui4 is compiled
 * against Gtk+ 2.4.0 or higher, as Gtk+ 2.4.0 has built-in support for icon
 * theming.
 *
 * Since 4.1
 **/
void
xfce_set_icon_theme(const gchar *theme_name)
{
#if !GTK_CHECK_VERSION(2, 4, 0)
	g_snprintf(icon_theme, 256, theme_name);
#endif
}

/*< deprecated name; use xfce_themed_icon_load() instead >*/
GdkPixbuf *
xfce_load_themed_icon(const gchar *name, gint size)
{
	return xfce_themed_icon_load(name, size);
}

/**
 * xfce_themed_icon_load:
 * @name: The name of the icon.
 * @size: The requested size of the icon.
 *
 * Uses the current icon theme to find an icon matching @name and @size.  For
 * themed icons, @name should be the root name of the icon, without extension.
 * #xfce_load_themed_icon() will also attempt to directly load icon files if an
 * absoulte or relative path is provided.  If an icon is found, it is resized
 * to the specified size if needed.
 *
 * Return value: A pointer to a #GdkPixbuf, or %NULL on failure.  This data must
 *               be freed with g_object_unref() after use.
 *
 * Since 4.1
 **/
GdkPixbuf *
xfce_themed_icon_load(const gchar *name, gint size)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	GdkPixbuf *pix = NULL;
	gchar *icon_path;
	
	g_return_val_if_fail(name && *name, NULL);
	
	icon_path = xfce_themed_icon_lookup(name, size);
	if(icon_path) {
		pix = xfce_pixbuf_new_from_file_at_size(icon_path, size, size, NULL);
		g_free(icon_path);
	}
	
	return pix;
#else
	return icon_find(name, size);
#endif
}

/**
 * xfce_themed_icon_lookup:
 * @name: The name of the icon.
 * @size: The requested size of the icon.
 *
 * Uses the current icon theme to find an icon matching @name and @size.  For
 * themed icons, @name should be the root name of the icon, without extension.
 * #xfce_themed_icon_lookup() will also attempt to directly load icon files if
 * an absoulte or relative path is provided.  
 *
 * Return value: An icon filename (should be freed with #g_free()), or %NULL.
 *
 * Since 4.1
 **/
gchar *
xfce_themed_icon_lookup(const gchar *name, gint size)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	GtkIconInfo *iinfo = NULL;
	gchar *p, *tmp, *tmp1;
	
	g_return_val_if_fail(name && *name, NULL);
	
	if(!icon_theme)
		ensure_icon_theme();
	
	if(*name == '/' && g_file_test(name, G_FILE_TEST_IS_REGULAR))
		return g_strdup(name);
	else if(g_strrstr(name, "/")) {
		/* GtkIconTheme doesn't like relative pathnames.  first try the current
		 * directory, then in $datadir/pixmaps.*/
		gchar *curdir = g_get_current_dir();
		tmp = g_build_filename(curdir, name, NULL);
		if(g_file_test(tmp, G_FILE_TEST_IS_REGULAR)) {
			g_free(curdir);
			return tmp;
		}
		g_free(curdir);
		g_free(tmp);
		
		/* not found; try elsewhere */
		tmp = g_build_filename(DATADIR, "pixmaps", name, NULL);
		if(g_file_test(tmp, G_FILE_TEST_IS_REGULAR))
			return tmp;
		g_free(tmp);
	} else {
		/* GtkIconTheme doesn't like icon names with extensions */
		tmp = g_strdup(name);
		if((p=g_strrstr(tmp, ".")))
			*p = 0;
		
		iinfo = gtk_icon_theme_lookup_icon(icon_theme, tmp, size, 0);
		if(!iinfo) {
			/* try some well-used sizes */
			iinfo = gtk_icon_theme_lookup_icon(icon_theme, tmp, 48, 0);
			if(!iinfo)
				iinfo = gtk_icon_theme_lookup_icon(icon_theme, tmp, 24, 0);
			if(!iinfo) {
				/* the icon may be oddly-sized, or in the current directory,
				 * and GtkIconTheme won't find it with it */
				gchar *curdir = g_get_current_dir();
				tmp1 = g_build_filename(curdir, name, NULL);
				if(g_file_test(tmp1, G_FILE_TEST_IS_REGULAR)) {
					g_free(curdir);
					return tmp1;
				}
				g_free(curdir);
				g_free(tmp1);
				
				/* not found; try elsewhere */
				tmp1 = g_build_filename(DATADIR, "pixmaps", name, NULL);
				if(g_file_test(tmp1, G_FILE_TEST_IS_REGULAR))
					return tmp1;
				g_free(tmp1);
			}
		}
		g_free(tmp);
	}

	if(iinfo) {
		tmp = g_strdup(gtk_icon_info_get_filename(iinfo));
		gtk_icon_info_free(iinfo);
		return tmp;
	}
	
	return NULL;
#else
	return icon_lookup(name);
#endif
}

void
xfce_themed_icon_add_search_path(const gchar *path)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	gchar **search_paths = NULL;
	gint n_elem = 0, i;
	
	ensure_icon_theme();
	
	gtk_icon_theme_get_search_path(icon_theme, &search_paths, &n_elem);
	for(i = 0; i < n_elem; i++) {
		if(!strcmp(search_paths[i], path)) {
			g_strfreev(search_paths);
			return;
		}
	}
	
	gtk_icon_theme_prepend_search_path(icon_theme, path);
	
	if(search_paths)
		g_strfreev(search_paths);
#else
	/* FIXME: IMPLEMENT ME! */
#endif
}

/**
 * xfce_pixbuf_new_from_file_at_size:
 * @filename: An image file name.
 * @width: The target image width.
 * @height: The target image height.
 * @error: a pointer to a GError, or NULL.
 *
 * Convenience function to avoid GTK_CHECK_VERSION(), as GdkPixbuf 2.2 and below
 * do not have gdk_pixbuf_new_from_file_at_size().
 *
 * Return value: A new GdkPixbuf, sized @width by @height, or NULL if the file
 *               cannot be opened.  If @error is not NULL, it will be set upon
 *               error.
 *
 * Since 4.1
 **/
GdkPixbuf *
xfce_pixbuf_new_from_file_at_size(const gchar *filename, gint width, gint height,
		GError **error)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return gdk_pixbuf_new_from_file_at_size(filename, width, height, error);
#else
	GdkPixbuf *pix, *tmp;
	gint w, h;
	
	pix = gdk_pixbuf_new_from_file(filename, error);
	if(pix) {
		w = gdk_pixbuf_get_width(pix);
		h = gdk_pixbuf_get_height(pix);
		if(w != width || h != height) {
			tmp = gdk_pixbuf_scale_simple(pix, width, height,
					GDK_INTERP_BILINEAR);
			g_object_unref(G_OBJECT(pix));
			pix = tmp;
		}
	}
	
	return pix;
#endif
}
