/*
 * Copyright © 2001 Red Hat, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Red Hat not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  Red Hat makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * RED HAT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL RED HAT
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author:  Owen Taylor, Red Hat, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 *          Olivier Fourdan: adapted to the "multi-channel" concept
 *          Benedikt Meurer: added mmap(2) support and reader/writer locking
 *                           for platforms that supports it. Some speedups
 *                           and bugfixes.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#include <errno.h>
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_MMAP
#include <sys/mman.h>
#endif

#include <X11/Xmd.h>

#include <glib.h>

#include <libxfce4util/i18n.h>

#include "mcs-channel.h"
#include "mcs-manager.h"

struct _McsManager
{
    Display *display;
    int screen;

    Window std_window;
    Atom std_manager_atom;
    Atom std_selection_atom;

    Window mcs_window;
    Atom mcs_manager_atom;
    Atom mcs_selection_atom;

    Atom show_atom;

    McsTerminateFunc terminate;
    McsShowRequestFunc show;
    void *cb_data;

    McsChannelList *channels;
    unsigned long serial;
};

McsChannelList *channels;
McsList *settings;

typedef struct
{
    Window window;
    Atom timestamp_prop_atom;
}
TimeStampInfo;

static Bool
timestamp_predicate(Display *display, XEvent *xevent, XPointer arg)
{
    TimeStampInfo *info = (TimeStampInfo *) arg;

    if(xevent->type == PropertyNotify &&
       xevent->xproperty.window == info->window && 
       xevent->xproperty.atom == info->timestamp_prop_atom)
        return True;

    return False;
}

/**
 * get_server_time:
 * @display: display from which to get the time
 * @window: a #Window, used for communication with the server.
 *          The window must have PropertyChangeMask in its
 *          events mask or a hang will result.
 * 
 * Routine to get the current X server time stamp. 
 * 
 * Return value: the time stamp.
 **/
static Time
get_server_time(Display *display, Window window)
{
    unsigned char c = 'a';
    XEvent xevent;
    TimeStampInfo info;

    info.timestamp_prop_atom = XInternAtom(display, "_TIMESTAMP_PROP", False);
    info.window = window;

    XChangeProperty(display, window, info.timestamp_prop_atom, info.timestamp_prop_atom, 8, PropModeReplace, &c, 1);

    XIfEvent(display, &xevent, timestamp_predicate, (XPointer) & info);

    return xevent.xproperty.time;
}

McsChannel *mcs_manager_add_channel(McsManager * manager, const gchar *channel_name)
{
    McsChannel *channel;
    McsChannelList *channels_list;
    McsChannelList *new_elemt;

    g_return_val_if_fail(channel_name != NULL, NULL);
    g_return_val_if_fail(manager != NULL, NULL);

    if((manager->std_selection_atom == None) && !g_ascii_strncasecmp(channel_name, "SETTINGS", strlen("SETTINGS")))
    {
        return NULL;
    }

    if ((channel = _mcs_channel_lookup(manager->channels,channel_name)) != NULL)
        return(channel);

    channel = _mcs_channel_new(channel_name, manager->display);
    channels_list = manager->channels;
    new_elemt = g_new(McsChannelList, 1);
    new_elemt->channel = channel;
    new_elemt->next = NULL;
    if(channels_list)
    {
        for (; channels_list->next != NULL; channels_list = channels_list->next)
            ;

        channels_list->next = new_elemt;
    }
    else
    {
        manager->channels = new_elemt;
    }

    return channel;
}

void
mcs_manager_delete_channel(McsManager *manager, const gchar *channel_name)
{
    McsChannel *channel;

    g_return_if_fail(channel_name != NULL);
    g_return_if_fail(manager != NULL);

    if ((channel = _mcs_channel_lookup(manager->channels,channel_name)) != NULL)
        _mcs_channel_delete(&manager->channels, channel);
}

static Bool mcs_manager_get_selection(McsManager * manager, Atom atom)
{
    XClientMessageEvent xev;
    Time timestamp;
    Window w;
    Atom a;

    g_return_val_if_fail(manager != NULL, False);
    
    if(atom == manager->mcs_selection_atom)
    {
        w = manager->mcs_window;
        a = manager->mcs_manager_atom;
    }
    else
    {
        w = manager->std_window;
        a = manager->std_manager_atom;
    }

    timestamp = get_server_time(manager->display, w);

    XSetSelectionOwner(manager->display, atom, w, timestamp);

    if(XGetSelectionOwner(manager->display, atom) == w)
    {
        xev.type = ClientMessage;
        xev.window = RootWindow(manager->display, manager->screen);
        xev.message_type = a;
        xev.format = 32;
        xev.data.l[0] = timestamp;
        xev.data.l[1] = atom;
        xev.data.l[2] = w;
        xev.data.l[3] = 0;      /* manager specific data */
        xev.data.l[4] = 0;      /* manager specific data */

        XSendEvent(manager->display, RootWindow(manager->display, manager->screen), False, StructureNotifyMask, (XEvent *) & xev);
        return True;
    }
    return False;
}

McsManager *mcs_manager_new(Bool std_mcs, Display * display, int screen, McsTerminateFunc terminate, McsShowRequestFunc show, void *cb_data)
{
    McsManager *manager;
    gchar *buffer;

    if ((manager = g_new(McsManager, 1)) == NULL)
        return(NULL);

    manager->display = display;
    manager->screen = screen;
    manager->mcs_manager_atom = XInternAtom(display, "MCS_MANAGER", False);
    manager->show_atom = XInternAtom(display, "SHOW", False);
    manager->terminate = terminate;
    manager->show = show;
    manager->cb_data = cb_data;
    manager->channels = NULL;
    manager->serial = 0;
    manager->mcs_window = XCreateSimpleWindow(display, RootWindow(display, screen), -10, -10, 10, 10, 0, WhitePixel(display, screen), WhitePixel(display, screen));

    XSelectInput(display, manager->mcs_window, PropertyChangeMask);

    buffer = g_strdup_printf("_MCS_S%d", screen);
    manager->mcs_selection_atom = XInternAtom(display, buffer, False);
    g_free(buffer);

    if(!mcs_manager_get_selection(manager, manager->mcs_selection_atom))
    {
        manager->terminate(manager->cb_data);
    }
    else if(std_mcs)
    {
        manager->std_window = XCreateSimpleWindow(display, RootWindow(display, screen), -10, -10, 10, 10, 0, WhitePixel(display, screen), WhitePixel(display, screen));
        XSelectInput(display, manager->std_window, PropertyChangeMask);
        manager->std_manager_atom = XInternAtom(display, "MANAGER", False);
        buffer = g_strdup_printf("_XSETTINGS_S%d", screen);
        manager->std_selection_atom = XInternAtom(display, buffer, False);
	g_free(buffer);
        if(!mcs_manager_get_selection(manager, manager->std_selection_atom))
        {
            manager->terminate(manager->cb_data);
        }
    }
    else
    {
        manager->std_selection_atom = None;
        manager->std_window = None;
    }

    return manager;
}

void
mcs_manager_destroy(McsManager *manager)
{
    g_return_if_fail(manager != NULL);

    if(manager->mcs_window)
        XDestroyWindow(manager->display, manager->mcs_window);

    if(manager->std_window)
        XDestroyWindow(manager->display, manager->std_window);

    while (manager->channels != NULL) {
        if (manager->channels->channel == NULL ||
                manager->channels->channel->channel_name == NULL)
            g_warning(_("Bogus MCS manager channels"));
        else
            _mcs_channel_delete(&manager->channels, manager->channels->channel);
    }

    g_free(manager);
}

Window mcs_manager_get_std_window(McsManager * manager)
{
    return manager->std_window;
}

Window mcs_manager_get_mcs_window(McsManager * manager)
{
    g_return_val_if_fail(manager != NULL, None);
    return manager->mcs_window;
}

Bool mcs_manager_process_event(McsManager * manager, XEvent * xev)
{
    g_return_val_if_fail(manager != NULL, False);
    
    if(xev->xany.window == manager->mcs_window && xev->xany.type == SelectionClear && xev->xselectionclear.selection == manager->mcs_selection_atom)
    {
        manager->terminate(manager->cb_data);
        return True;
    }
    else if(xev->xany.window == manager->std_window && xev->xany.type == SelectionClear && xev->xselectionclear.selection == manager->std_selection_atom)
    {
        manager->terminate(manager->cb_data);
        return True;
    }
    else if(xev->xany.window == manager->mcs_window && xev->xany.type == PropertyNotify && xev->xclient.message_type == manager->show_atom)
    {
        Atom type;
        int format;
        unsigned long n_items;
        unsigned long bytes_after;
        guchar *data;
        int result;

        result = XGetWindowProperty(manager->display, manager->mcs_window, manager->show_atom, 0, LONG_MAX, False, manager->show_atom, &type, &format, &n_items, &bytes_after, &data);
        if(result == Success)
        {
            if((type == manager->show_atom) && (manager->show))
            {
                manager->show(data, manager->cb_data);
            }
            XFree(data);
            return True;
        }
    }


    return False;
}

McsResult mcs_manager_set_setting(McsManager * manager, McsSetting * setting, const gchar *channel_name)
{
    McsChannel *channel;
    McsSetting *old_setting;
    McsSetting *new_setting;
    McsResult result;

    g_return_val_if_fail (manager != NULL, MCS_FAILED);
    g_return_val_if_fail (channel_name != NULL, MCS_FAILED);
    g_return_val_if_fail (setting != NULL, MCS_FAILED);

    if ((channel = _mcs_channel_lookup(manager->channels,channel_name)) == NULL)
        return(MCS_NO_CHANNEL);

    old_setting = mcs_list_lookup(channel->settings, setting->name);

    if(old_setting)
    {
        if(mcs_setting_equal(old_setting, setting))
            return MCS_SUCCESS;

        mcs_list_delete(&channel->settings, setting->name);
    }

    new_setting = mcs_setting_copy(setting);
    if(!new_setting)
        return MCS_NO_MEM;

    new_setting->last_change_serial = manager->serial;

    result = mcs_list_insert(&channel->settings, new_setting);

    if(result != MCS_SUCCESS)
        mcs_setting_free(new_setting);

    return result;
}

McsResult mcs_manager_delete_setting(McsManager * manager, const gchar *name, const gchar *channel_name)
{
    McsChannel *channel;

    g_return_val_if_fail (manager != NULL, MCS_FAILED);
    g_return_val_if_fail (channel_name != NULL, MCS_FAILED);
    g_return_val_if_fail (name != NULL, MCS_FAILED);

    if ((channel = _mcs_channel_lookup(manager->channels,channel_name)) != NULL)
        return(mcs_list_delete(&channel->settings, name));

    return(MCS_NO_CHANNEL);
}

McsResult mcs_manager_set_int(McsManager * manager, const gchar *name, const gchar *channel_name, int value)
{
    McsSetting setting;

    g_return_val_if_fail (manager != NULL, MCS_FAILED);
    g_return_val_if_fail (channel_name != NULL, MCS_FAILED);
    g_return_val_if_fail (name != NULL, MCS_FAILED);

    setting.name = (gchar *)name;
    setting.channel_name = (gchar *)channel_name;
    setting.type = MCS_TYPE_INT;
    setting.data.v_int = value;

    return mcs_manager_set_setting(manager, &setting, channel_name);
}

McsResult mcs_manager_set_string(McsManager * manager, const gchar *name, const gchar *channel_name, const gchar *value)
{
    McsSetting setting;

    g_return_val_if_fail (manager != NULL, MCS_FAILED);
    g_return_val_if_fail (channel_name != NULL, MCS_FAILED);
    g_return_val_if_fail (name != NULL, MCS_FAILED);
    g_return_val_if_fail (value != NULL, MCS_FAILED);

    setting.name = (gchar *)name;
    setting.channel_name = (gchar *)channel_name;
    setting.type = MCS_TYPE_STRING;
    setting.data.v_string = (gchar *)value;

    return mcs_manager_set_setting(manager, &setting, channel_name);
}

McsResult mcs_manager_set_color(McsManager * manager, const gchar *name, const gchar *channel_name, McsColor * value)
{
    McsSetting setting;

    g_return_val_if_fail (manager != NULL, MCS_FAILED);
    g_return_val_if_fail (channel_name != NULL, MCS_FAILED);
    g_return_val_if_fail (name != NULL, MCS_FAILED);
    g_return_val_if_fail (value != NULL, MCS_FAILED);

    setting.name = (gchar *)name;
    setting.channel_name = (gchar *)channel_name;
    setting.type = MCS_TYPE_COLOR;
    setting.data.v_color = *value;

    return mcs_manager_set_setting(manager, &setting, channel_name);
}

static size_t
setting_length(const McsSetting * setting)
{
    size_t length = 8;          /* type + pad + name-len + last-change-serial */
    length += MCS_PAD(strlen(setting->name), 4);

    switch (setting->type)
    {
        case MCS_TYPE_INT:
            length += 4;
            break;
        case MCS_TYPE_STRING:
            length += 4 + MCS_PAD(strlen(setting->data.v_string), 4);
            break;
        case MCS_TYPE_COLOR:
            length += 8;
            break;
    }

    return length;
}

static void
setting_store(const McsSetting * setting, McsBuffer * buffer)
{
    size_t string_len;
    size_t length;

    *(buffer->pos++) = setting->type;
    *(buffer->pos++) = 0;

    string_len = strlen(setting->name);
    *(CARD16 *) (buffer->pos) = string_len;
    buffer->pos += 2;

    length = MCS_PAD(string_len, 4);
    memcpy(buffer->pos, setting->name, string_len);
    length -= string_len;
    buffer->pos += string_len;

    while(length > 0)
    {
        *(buffer->pos++) = 0;
        length--;
    }

    *(CARD32 *) (buffer->pos) = setting->last_change_serial;
    buffer->pos += 4;

    switch (setting->type)
    {
        case MCS_TYPE_INT:
            *(CARD32 *) (buffer->pos) = setting->data.v_int;
            buffer->pos += 4;
            break;
        case MCS_TYPE_STRING:
            string_len = strlen(setting->data.v_string);
            *(CARD32 *) (buffer->pos) = string_len;
            buffer->pos += 4;

            length = MCS_PAD(string_len, 4);
            memcpy(buffer->pos, setting->data.v_string, string_len);
            length -= string_len;
            buffer->pos += string_len;

            while(length > 0)
            {
                *(buffer->pos++) = 0;
                length--;
            }
            break;
        case MCS_TYPE_COLOR:
            *(CARD16 *) (buffer->pos) = setting->data.v_color.red;
            *(CARD16 *) (buffer->pos + 2) = setting->data.v_color.green;
            *(CARD16 *) (buffer->pos + 4) = setting->data.v_color.blue;
            *(CARD16 *) (buffer->pos + 6) = setting->data.v_color.alpha;
            buffer->pos += 8;
            break;
    }
}

McsResult mcs_manager_notify(McsManager * manager, const gchar *channel_name)
{
    McsChannel *channel;
    McsBuffer buffer;
    McsList *iter;
    int n_settings = 0;

    g_return_val_if_fail (manager != NULL, MCS_FAILED);
    g_return_val_if_fail (channel_name != NULL, MCS_FAILED);

    if ((channel = _mcs_channel_lookup(manager->channels,channel_name)) == NULL)
        return(MCS_NO_CHANNEL);

    buffer.len = 12;            /* byte-order + pad + SERIAL + N_SETTINGS */

    iter = channel->settings;
    while(iter)
    {
        buffer.len += setting_length(iter->setting);
        n_settings++;
        iter = iter->next;
    }

    buffer.data = buffer.pos = g_malloc(buffer.len);
    if(!buffer.data)
    {
        return MCS_NO_MEM;
    }
    *buffer.pos = mcs_byte_order();

    buffer.pos += 4;
    *(CARD32 *) buffer.pos = manager->serial++;
    buffer.pos += 4;
    *(CARD32 *) buffer.pos = n_settings;
    buffer.pos += 4;

    iter = channel->settings;
    while(iter)
    {
        setting_store(iter->setting, &buffer);
        iter = iter->next;
    }

    if(!g_ascii_strncasecmp(channel_name, "SETTINGS", strlen("SETTINGS")))
    {
        if(manager->std_window)
        {
            XChangeProperty(manager->display, manager->std_window, channel->channel_atom, channel->channel_atom, 8, PropModeReplace, buffer.data, buffer.len);
        }
    }
    else
    {
        XChangeProperty(manager->display, manager->mcs_window, channel->channel_atom, channel->channel_atom, 8, PropModeReplace, buffer.data, buffer.len);
    }

    g_free(buffer.data);

    return MCS_SUCCESS;
}

McsList *
mcs_manager_list_lookup(McsManager *manager, const gchar *channel_name)
{
    McsChannel *channel;

    g_return_val_if_fail (manager != NULL, NULL);
    g_return_val_if_fail (channel_name != NULL, NULL);

    if ((channel = _mcs_channel_lookup(manager->channels,channel_name)) != NULL)
        return(channel->settings);

    return(NULL);
}

McsSetting *
mcs_manager_setting_lookup(McsManager *manager, const gchar *name,
        const gchar *channel_name)
{
    McsList *list;
    
    g_return_val_if_fail (manager != NULL, NULL);
    g_return_val_if_fail (name != NULL, NULL);
    g_return_val_if_fail (channel_name != NULL, NULL);

    
    if ((list = mcs_manager_list_lookup(manager, channel_name)) != NULL)
        return(mcs_list_lookup(list, name));

    return(NULL);
}

/* Load/Save options helper functions */

typedef enum
{
    START,
    MCS_OPTION,
    OPTION,
    UNKNOWN
}
ParserState;

typedef struct _McsOptionParser McsOptionParser;
struct _McsOptionParser
{
    const gchar *filename;
    const gchar *channel_name;
    McsManager *manager;
    ParserState state;
};

static McsResult
option_to_mcs_setting(McsManager *manager, const gchar *channel_name,
        const gchar *name, const gchar *type, const gchar *value)
{
    if((!name) || !strlen(name))
    {
        g_warning(_("Missing name"));
        return MCS_FAILED;
    }
    else if((!type) || !strlen(type))
    {
        g_warning(_("Missing type"));
        return MCS_FAILED;
    }
    else if(!value)
    {
        g_warning(_("Missing value"));
        return MCS_FAILED;
    }

    if(!strcmp(type, "string"))
    {
        return mcs_manager_set_string(manager, name, channel_name, value);
    }
    else if(!strcmp(type, "int"))
    {
        return mcs_manager_set_int(manager, name, channel_name, atoi(value));
    }
    else if(!strcmp(type, "color"))
    {
        McsColor color;
        unsigned int r, g, b, a;
	
        sscanf(value, "%u,%u,%u,%u", &r, &g, &b, &a);
	    color.red   = (guint16) r;
	    color.green = (guint16) g;
	    color.blue  = (guint16) b;
	    color.alpha = (guint16) a;

        return mcs_manager_set_color(manager, name, channel_name, &color);
    }
    g_warning(_("Invalid type \"%s\""), type);
    return MCS_FAILED;
}

static void
start_element_handler(GMarkupParseContext *context, const gchar *element_name,
        const gchar **attribute_names, const gchar **attribute_values,
        gpointer user_data, GError **error)
{
    McsOptionParser *parser = (McsOptionParser *) user_data;

    switch (parser->state)
    {
        case START:
            if(strcmp(element_name, "mcs-option") == 0)
                parser->state = MCS_OPTION;
            break;
        case MCS_OPTION:
            if(strcmp(element_name, "option") == 0)
            {
                int i = 0;
                gchar *name = NULL;
                gchar *type = NULL;
                gchar *value = NULL;

                while(attribute_names[i] != NULL)
                {
                    if(!strcmp(attribute_names[i], "name"))
                    {
                        name = (gchar *) attribute_values[i];
                    }
                    else if(!strcmp(attribute_names[i], "type"))
                    {
                        type = (gchar *) attribute_values[i];
                    }
                    else if(!strcmp(attribute_names[i], "value"))
                    {
                        value = (gchar *) attribute_values[i];
                    }
                    ++i;
                }
                if(name && type && value)
                {
                    option_to_mcs_setting(parser->manager, parser->channel_name, name, type, value);
                }
                else
                {
                    g_warning(_("missing data"));
                }
            }
            break;
        default:
            g_warning(_("start unknown element \"%s\""), element_name);
            break;
    }
}

static void
end_element_handler(GMarkupParseContext *context, const gchar *element_name,
        gpointer user_data, GError **error)
{
    McsOptionParser *parser = (McsOptionParser *) user_data;

    switch (parser->state)
    {
        case START:
            g_warning(_("parser: This shouldn't happen.\n"));
            break;
        case MCS_OPTION:
            if(strcmp(element_name, "mcs-option") == 0)
                parser->state = START;
            break;
        default:
            g_warning(_("end unknown element \"%s\""), element_name);
            break;
    }
}

static void
error_handler(GMarkupParseContext *context, GError *error, gpointer user_data)
{
    if ((error) && (error->message))
    {
        g_warning(" %s", error->message);
    }
}

static GMarkupParser markup_parser = {
    start_element_handler,
    end_element_handler,
    NULL,
    NULL,
    error_handler
};

McsChannel*
mcs_manager_add_channel_from_file (McsManager  *manager,
                                   const gchar *channel_name,
                                   const gchar *filename)
{
  gchar *contents;
  GError *error;
  GMarkupParseContext *context;
  McsOptionParser parser;
  McsChannel *channel;
  struct stat sb;
#ifdef HAVE_MMAP
  void *addr;
#endif
  size_t bytes;
  int fd, rc;

  g_return_val_if_fail (manager != NULL, NULL);
  g_return_val_if_fail (filename != NULL && strlen (filename) > 0, NULL);
  g_return_val_if_fail (channel_name != NULL && strlen (channel_name) > 0, NULL);

  channel = mcs_manager_add_channel (manager, channel_name);
  if (channel == NULL)
    {
      g_critical ("Unable to add channel \"%s\" to MCS manager",
		  channel_name);
      return NULL;
    }
    
  if (stat (filename, &sb) < 0)
    {
#if 0
      /* The file doesn't exists, so what, it's not "critical"! */
      g_critical ("Unable to stat file %s to load data of channel \"%s\": %s",
		  filename,
                  channel_name,
		  g_strerror (errno));
#endif
      return NULL;
    }

  if ((fd = open (filename, O_RDONLY, 0)) < 0)
    {
      g_critical ("Unable to open file %s to load data of channel \"%s\": %s",
		  filename,
		  channel_name,
		  g_strerror (errno));
      return NULL;
    }

#ifdef HAVE_MMAP
  /* Try to mmap(2) the config file, as this save us a lot of
   * kernelspace -> userspace copying
   */
#ifdef MAP_FILE
  addr = mmap (NULL, sb.st_size, PROT_READ, MAP_FILE | MAP_PRIVATE, fd, 0);
#else
  addr = mmap (NULL, sb.st_size, PROT_READ, MAP_PRIVATE, fd, 0);
#endif

  if (addr != NULL)
    {
      /* nice, mmap did the job */
      contents = addr;
    }
  else
    {
      g_warning ("Failed to mmap file %s to load data of channel \"%s\": %s. "
                 "Using read fallback.",
                 filename,
                 channel_name,
                 g_strerror (errno));
#endif
      contents = g_malloc ((size_t)sb.st_size);
      if (contents == NULL)
        {
          g_critical ("Unable to allocate %lu bytes of memory to load contents "
                      "of file %s for channel \"%s\": %s",
                      (unsigned long)sb.st_size,
                      filename,
                      channel_name,
                      g_strerror (errno));
          goto finished;
        }

      for (bytes = 0; bytes < (size_t)sb.st_size; )
        {
          errno = 0;
          rc = read (fd, contents + bytes, sb.st_size - bytes);
          
          if (rc < 0)
            {
              if (errno == EINTR || errno == EAGAIN)
                continue;
              
              g_critical ("Unable to read contents from file %s: %s",
                          filename,
                          g_strerror (errno));
              goto finished2;
            }
          else if (rc == 0)
            {
              g_critical ("Unexpected end of file reading contents from "
                          "file %s: %s",
                          filename,
                          g_strerror (errno));
            }
          
          bytes += rc;
        }
      
#ifdef HAVE_MMAP
    }
#endif

  error = NULL;

  parser.state = START;
  parser.filename = filename;
  parser.manager = manager;
  parser.channel_name = channel_name;

  context = g_markup_parse_context_new (&markup_parser, 0, &parser, NULL);

  if (!g_markup_parse_context_parse (context, contents, sb.st_size, &error))
    {
      g_critical ("Unable to parse file %s into channel \"%s\": %s",
		  filename,
		  channel_name,
		  error->message);
      g_error_free (error);
      goto finished3;
    }

  if (!g_markup_parse_context_end_parse (context, &error))
    {
      g_critical ("Unable to parse file %s into channel \"%s\": %s",
		  filename,
		  channel_name,
		  error->message);
      g_error_free (error);
      goto finished3;
    }

  mcs_manager_notify (manager, channel_name);

 finished3:
  g_markup_parse_context_free (context);

 finished2:
#ifdef HAVE_MMAP
  if (addr != NULL)
    {
      if (munmap (addr, sb.st_size) < 0)
        {
          g_critical ("Unable to unmap file %s with contents for channel "
                      "\"%s\": %s. This should not happen!",
                      filename,
                      channel_name,
                      g_strerror (errno));
        }
    }
  else
#endif
    g_free (contents);

 finished:
  if (close (fd) < 0)
    {
      g_critical ("Failed to close file %s: %s",
		  filename,
		  g_strerror (errno));
    }

  return channel;
}

gboolean
mcs_manager_save_channel_to_file (McsManager  *manager,
                                  const gchar *channel_name,
                                  const gchar *filename)
{
  McsSetting *setting;
  McsList *iter;
  McsList *list;
  FILE *fp;
  gchar tmp_path[PATH_MAX];
    
  g_return_val_if_fail (manager != NULL, FALSE);
  g_return_val_if_fail (filename != NULL
			|| (strlen(filename) > 0), FALSE);
  g_return_val_if_fail (channel_name != NULL
			|| (strlen(channel_name) > 0), FALSE);

  g_snprintf (tmp_path, PATH_MAX, "%s.tmp", filename);

  fp = fopen (tmp_path, "w");
  if (fp == NULL)
    {
      g_critical ("Unable to open file %s to store channel \"%s\" to: %s",
		  tmp_path,
		  channel_name,
		  g_strerror (errno));
      return FALSE;
    }

  /* Write header */
  fprintf (fp,
	   "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
	   "<!DOCTYPE mcs-option SYSTEM \"mcs-option.dtd\">\n"
	   "\n"
	   "<mcs-option>\n");

  list = mcs_manager_list_lookup (manager, channel_name);

  for (iter = list; iter != NULL; iter = iter->next)
    {
      setting = iter->setting;
           
      switch (setting->type) {
      case MCS_TYPE_INT:
	fprintf (fp,
		 "\t<option name=\"%s\" type=\"int\" value=\"%i\"/>\n",
		 setting->name, setting->data.v_int);
	break;
		
      case MCS_TYPE_COLOR:
	fprintf (fp, "\t<option name=\"%s\" type=\"color\" "
		 "value=\"%16u,%16u,%16u,%16u\"/>\n",
		 setting->name,
		 setting->data.v_color.red,
		 setting->data.v_color.green,
		 setting->data.v_color.blue,
		 setting->data.v_color.alpha);
	break;
	        
      case MCS_TYPE_STRING:
	fprintf (fp, "\t<option name=\"%s\" type=\"string\" "
		 "value=\"%s\"/>\n",
		 setting->name,
		 setting->data.v_string);
	break;

      default:
	break;
      }
    }

  fprintf (fp, "</mcs-option>\n");

  if (fclose (fp) == EOF)
    {
      g_critical ("Unable to close file handle for %s: %s",
		  tmp_path,
		  g_strerror (errno));
      unlink (tmp_path);
      return FALSE;
    }

  if (rename (tmp_path, filename) < 0)
    {
      g_critical ("Unable to rename file %s to %s: %s",
		  tmp_path,
		  filename,
		  g_strerror (errno));
      return FALSE;
    }

  return TRUE;
}
