/*
 * Copyright (c) 2004 Jean-Franois Wauthy (pollux@xfce.org)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtkversion.h>
#if GTK_CHECK_VERSION(2, 4, 0)
#undef GTK_DISABLE_DEPRECATED
#endif

#include <gmodule.h>
#include <string.h>
#include <libxfcegui4/libxfcegui4.h>
#include <libxfce4util/libxfce4util.h>
#include <printing-system.h>

static GList *printers = NULL;

/***************/
/* Main window */
/***************/
static GList *get_printers_list ()
{
  GList *list = NULL;
  FILE *printcap;
  gchar buf[80];

  /* load printer list from /etc/printcap */
  printcap = fopen ("/etc/printcap", "r-");
	
  if (printcap) {
    int i;

    while (fgets (buf, sizeof (buf), printcap)) {
      if (buf[0] != '#') {
	gchar *printer_name = NULL;

	for (i=0; i<strlen(buf) && buf[i] != '|' && buf[i] != ':'; i++);
	printer_name = g_strndup (buf, i);
	g_strstrip (printer_name);
	if (strlen (printer_name) != 0) {
	  Printer *printer;
	  printer = g_new0 (Printer, 1);
	  
	  printer->name = g_strdup (printer_name);
	  printer->alias = g_strdup (printer_name); //TODO extract alias

	  list = g_list_append (list, printer);
	}
	g_free (printer_name);
      }
    }
    fclose(printcap);
  }
  
  return list;
}

/* Returns the printer list  */
/* returned pointer must be freed */
G_MODULE_EXPORT
GList* printing_system_get_printers_list_impl ()
{
  return get_printers_list ();
}

/* Customize main window */
G_MODULE_EXPORT
void printing_system_main_window_customize_impl (ManagerDlg *dlg)
{
  gtk_widget_set_sensitive (dlg->menuitem_properties, FALSE);
  gtk_widget_set_sensitive (dlg->menuitem_setdefault, FALSE);
}

G_MODULE_EXPORT
gboolean printing_system_set_default_printer_impl (const gchar *printer)
{
  return TRUE;
}

G_MODULE_EXPORT
Printer *printing_system_get_default_printer_impl (GList *printers)
{
  return NULL;
}
/****************/
/* Queue dialog */
/****************/

/* Refresh the job list */
/* returned pointer must be freed */
G_MODULE_EXPORT
GList *printing_system_get_jobs_list_impl (const gchar *printer)
{
  return NULL;
}

/* Remove job from printer_name queue */
G_MODULE_EXPORT
gboolean printing_system_remove_job_impl (const gchar *printer, gint id)
{
  return TRUE;
}

/* Customize the queue dialog */
G_MODULE_EXPORT
void printing_system_queue_dialog_customize_impl (QueueDlg *dialog)
{
  gtk_widget_set_sensitive (dialog->treeview, FALSE);
  gtk_widget_set_sensitive (dialog->menuitem_removejob, FALSE);
  gtk_widget_set_sensitive (dialog->menuitem_refresh, FALSE);
}

/*********************/
/* Properties dialog */
/*********************/
/* Show the properties dialog */
G_MODULE_EXPORT
void printing_system_properties_dialog_run_impl (GtkWidget *widget, const gchar *printer)
{
}

/****************/
/* Print dialog */
/****************/
static gboolean destroy_print_dialog_cb (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
  g_list_foreach (printers, (GFunc)printer_free, NULL);
  g_list_free (printers);

  return FALSE;
}

/* Customize the print dialog */
G_MODULE_EXPORT
void printing_system_print_dialog_customize_impl (PrintDlg *dialog)
{
  GList *printer_el;
  GList *list = NULL;
  Printer *printer;
  GtkWidget *hbox;
  GtkWidget *label;
  
  printers = get_printers_list ();

  g_signal_connect (G_OBJECT(dialog->dialog), "destroy_event",
		    G_CALLBACK(destroy_print_dialog_cb), printers);

  printer_el = g_list_first (printers);
  while (printer_el) {
    printer = printer_el->data;

    list = g_list_append (list, printer->alias);

    printer_el = g_list_next (printer_el);
  }

  hbox = gtk_hbox_new(FALSE, 5);

  label = gtk_label_new (_("Print to :"));
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);
  dialog->customization = (gpointer) gtk_combo_new ();
  gtk_box_pack_start (GTK_BOX (hbox), GTK_WIDGET (dialog->customization), TRUE, TRUE, 0);

  gtk_combo_set_popdown_strings (GTK_COMBO (dialog->customization), list);
  gtk_combo_set_value_in_list (GTK_COMBO (dialog->customization), TRUE, FALSE);

  /* TODO select default printer */

  gtk_widget_show_all (hbox);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog->dialog)->vbox), hbox,
		      TRUE, TRUE, 0);

  gtk_box_reorder_child (GTK_BOX (GTK_DIALOG (dialog->dialog)->vbox), hbox, 0);

  g_list_free (list);
}

/****************/
/* About dialog */
/****************/
G_MODULE_EXPORT
void printing_system_about_dialog_run_impl (GtkWidget *widget, gpointer data)
{
  XfceAboutInfo *info;
  GtkWidget *dialog;
  GdkPixbuf *icon = NULL;
  ManagerDlg *dlg;

  dlg = (ManagerDlg *) data;

  info = xfce_about_info_new ("xfprint-bsdlpr-plugin", VERSION,
			      _("BSD-LPR plugin for Xfprint"),
			      XFCE_COPYRIGHT_TEXT("2004", "Xfce4 Team"),
			      XFCE_LICENSE_BSD);
  xfce_about_info_set_homepage (info, "http://www.xfce.org/");

  /* Credits */
  xfce_about_info_add_credit (info,
			      "Jean-Francois Wauthy",
			      "pollux@xfce.org",
			      _("Developer"));
  xfce_about_info_add_credit (info,
			      "Francois Le Clainche",
			      "fleclainche@wanadoo.fr",
			      _("Icon designer"));

  //  icon = xfce_themed_icon_load ("xfprint-bsdlpr-plugin" , 48);

  dialog = xfce_about_dialog_new (GTK_WINDOW (dlg->window), info, icon);

  gtk_window_set_default_size (GTK_WINDOW (dialog),500,400);
  xfce_about_info_free (info);

  gtk_dialog_run (GTK_DIALOG (dialog));
  
  gtk_widget_destroy (dialog);

  //  g_object_unref (icon);
}

/************/
/* Printing */
/************/
G_MODULE_EXPORT
gboolean printing_system_print_file_impl (PrintDlg *dialog, const gchar *file)
{
  gchar *cmd = NULL;
  gchar *base_name = NULL;
  Printer *printer = NULL;
  gboolean ret = FALSE;

  printer = printer_lookup_byalias (printers, gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (dialog->customization)->entry)));

  base_name = g_path_get_basename (dialog->original_name);

  cmd = g_strdup_printf ("lpr -P %s -T %s %s", printer->name, base_name, file);
  ret = xfce_exec_sync (cmd, FALSE, FALSE, NULL);

  g_free (cmd);
  g_free (base_name);

  return ret;
}

/**********/
/* Others */
/**********/

G_MODULE_EXPORT
void printing_system_info_init_impl (PrintingSystemInfo * psinfo)
{
  psinfo->name = g_strdup ("BSD-LPR");
  psinfo->description = g_strdup ("BSD-LPR printing system support for Xfprint");
  psinfo->version = g_strdup (VERSION);
  psinfo->author = g_strdup ("Jean-Francois Wauthy");
  psinfo->homepage = g_strdup ("http://www.xfce.org");
}
